<?php
/**
 * BwPostman Newsletter Component
 *
 * BwPostman helper class for backend.
 *
 * @version 1.0.8.2 bwpm
 * @package BwPostman-Admin
 * @author Romana Boldt
 * @copyright (C) 2012-2015 Boldt Webservice <forum@boldt-webservice.de>
 * @support http://www.boldt-webservice.de/forum/bwpostman.html
 * @license GNU/GPL, see LICENSE.txt
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

defined ('_JEXEC') or die ();

// Component name amd database prefix
if (!defined ('BWPOSTMAN_COMPONENT_NAME')) define ('BWPOSTMAN_COMPONENT_NAME', basename (dirname (__FILE__)));
if (!defined ('BWPOSTMAN_NAME')) define ('BWPOSTMAN_NAME', substr (BWPOSTMAN_COMPONENT_NAME, 4));

// Component location
if (!defined ('BWPOSTMAN_COMPONENT_LOCATION')) define ('BWPOSTMAN_COMPONENT_LOCATION', basename (dirname (dirname (__FILE__))));

// Component paths
if (!defined ('BWPOSTMAN_PATH_COMPONENT_RELATIVE')) define ('BWPOSTMAN_PATH_COMPONENT_RELATIVE', BWPOSTMAN_COMPONENT_LOCATION . '/' . BWPOSTMAN_COMPONENT_NAME);
if (!defined ('BWPOSTMAN_PATH_SITE')) define ('BWPOSTMAN_PATH_SITE', JPATH_ROOT .'/'. BWPOSTMAN_PATH_COMPONENT_RELATIVE);
if (!defined ('BWPOSTMAN_PATH_ADMIN')) define ('BWPOSTMAN_PATH_ADMIN', JPATH_ADMINISTRATOR .'/'. BWPOSTMAN_PATH_COMPONENT_RELATIVE);
if (!defined ('BWPOSTMAN_PATH_MEDIA')) define ('BWPOSTMAN_PATH_MEDIA', JPATH_ROOT .'/media/' . BWPOSTMAN_NAME);


abstract class BwPostmanHelper {
	static $session = null;

	/**
	 * Check if BwPostman is safe to be used.
	 *
	 * If installer is running, it's unsafe to use our framework. Files may be currently replaced with
	 * new ones and the database structure might be inconsistent. Using forum during installation will
	 * likely cause fatal errors and data corruption if you attempt to update objects in the database.
	 *
	 * Always detect BwPostman in your code before you start using the framework:
	 *
	 * <code>
	 *	// Check if BwPostman has been installed and compatible with your code
	 *	if (class_exists('BwPostmanAdmin') && BwPostmanHelper::installed() && BwPostmanHelper::isCompatible('2.0.0-BETA2')) {
	 *		// Initialize the framework (new in 2.0.0-BETA2)
	 *		BwPostmanForum::setup();
	 *		// Start using the framework
	 *	}
	 * </code>
	 *
	 * @see BwPostmanHelper::enabled()
	 * @see BwPostmanHelper::isCompatible()
	 * @see BwPostmanHelper::setup()
	 *
	 * @return boolean True.
	 */
	public static function installed() {
		return true;
	}


	/**
	 * Method to replace the links in a newsletter to provide the correct preview
	 *
	 * @access	public
	 * @param 	string HTML-/Text-version
	 * @return 	boolean
	 */
	static public function replaceLinks(&$text)
	{
		$search_str = '/\s+(href|src)\s*=\s*["\']?\s*(?!http|mailto)([\w\s&%=?#\/\.;:_-]+)\s*["\']?/i';
		$text = preg_replace($search_str,' ${1}="'.JURI::root().'${2}"',$text);
		return true;
	}
	
	/**
	 * Method to get selectlist for dates
	 *
	 * @access	public
	 * @param 	string				sort of date --> day, hour, minute
	 * @param 	int					length of listarray
	 * @param 	array of objects	selectval selected values
	 * @return 	string				selectlist
	 */
	public function getDateList($date = 'minute', $length = 10, $selectval)
	{
		$options	= array();
		$optMax		= 0;
		$selectlist	= array();
		$intval		= 1;
		if ($date == 'minute') {
			$intval = JComponentHelper::getParams('Com_bwpostman')->getValue('autocam_minute_intval') ;
		}

		switch ($date) {
			case 'day':		for ($i = 0; $i <= 31; $i++) {
								$options[] = $i;
							}
							break;

			case 'hour':	for ($i = 0; $i < 24; $i++) {
								$options[] = $i;
							}
							break;

			case 'minute':	for ($i = 0; $i < 60; $i += $intval) {
								$options[] = $i;
							}
							break;
		}
		$optMax = count($options);

		foreach ($selectval->$date as $key => $value) {
			$attribs	= 'class="inputbox" size="1"';
			$opt		= "automailing_values[" . $date . "][".$key."]";
			if ($value != '0') {
				$selected	= $value;
			}
			else {
				$selected	= 0;
			}
				
			$select_html		= '<select id="' . $opt . '" name="automailing_values['.$date.'][]" >';
			foreach ($options as $key => $value) {

				$select_html		.= '<option value="' . $key*$intval . '"';
				if ($selected == $key*$intval) $select_html		.= ' selected="selected"';
				$select_html		.= '>' . $value . '</option>';
			}
			$select_html		.= '</select>';
			$selectlist[]	= $select_html;
		}
		return $selectlist;
	}

	/**
	 * Gets a list of the actions that can be performed.
	 *
	 * @param	int		The asset ID.
	 * @param	int		The section ID.
	 *
	 * @return	JObject
	 */

	public static function getActions($assetId = 0, $context = '')
    {
        $user	= JFactory::getUser();
        $result	= new JObject;

        if (empty($assetId) && empty($context)) {
            $assetName = 'com_bwpostman';
            $actions = array('core.admin', 'core.manage', 'core.create', 'core.edit', 'core.delete', 'core.edit.state', 'core.edit.own', 'core.archive');
        }
        elseif (empty($context)) {
            $assetName = 'com_bwpostman.'.(int) $assetId;
            $actions = array('core.admin', 'core.manage', 'core.create', 'core.edit', 'core.delete', 'core.archive', 'core.edit.state');
        }
        else {
        	$assetName = 'com_bwpostman.'. $context;
        	if ($assetId != 0) {
        		$assetName = $assetName . '.' . (int) $assetId;
        		
        	}
        	if ($context == 'archive') {
        		$actions = array('core.admin', 'core.restore', 'core.edit', 'core.delete', 'core.edit.state', 'core.archive');
        	}
            elseif (($context == 'newsletter') || ($context == 'newsletters')) {
        		$actions = array('core.admin', 'core.manage', 'core.create', 'core.edit', 'core.send', 'core.delete', 'core.edit.state', 'core.archive');
        	}
        	else {
        		$context	= 'core';
        		$actions	= array($context . '.admin', $context . '.manage', $context . '.create', $context . '.edit', $context . '.delete', $context . '.archive', $context . '.edit.state');
        	}
        }
		$assetName = 'com_bwpostman';
        
        foreach ($actions as $action)
        {
            $result->set($action, $user->authorise($action, $assetName));
        }

        return $result;
    }

    /**
     * Method to check if you can archive an existing record.
     *
     * @param    int		$recordId	The record to test.
     * @param    int		$ownerId	The user to test against.
     * @param    string		$context    The name of the context.
     *
     * @return    boolean
     * @since    1.6
     */
    public static function allowArchive($recordId = 0, $ownerId = 0, $context = '')
    {
		// Initialise variables.
		$recordId	= (int) $recordId;
		$ownerId	= (int) $ownerId;
		$user		= JFactory::getUser();
		$userId		= $user->get('id');
		
		// Check general component archive permission first.
		if ($user->authorise('core.archive', 'com_bwpostman.'.$context.'.'. $recordId)) {
			return true;
		}

        // Then check context archive permission.
		if ($user->authorise('core.archive', 'com_bwpostman.' . $context)) {
			return true;
		}

        // Next check item archive permission.
		if ($user->authorise('core.archive', 'com_bwpostman.' . $context . '.' . $recordId)) {
			return true;
		}

		// Fallback on edit.own (only at context newsletter).
		if  ($context = 'newsletter') {
			// First test if the permission is available.
			if ($user->authorise('core.edit.own', 'com_bwpostman.'.$context . '.' . $recordId)) {
				// Test if the owner matches 'me'.
				if ($ownerId == $userId) return true;
			}
		}
		return false;
	}
 
   /**
     * Method to check if you can delete an archived record.
     *
     * @param    int		$recordId	The record to test.
     * @param    int		$ownerId	The user to test against.
     * @param    string		$context    The name of the context.
     *
     * @return    boolean
     * @since    1.6
     */
    public static function allowDelete($recordId = 0, $ownerId = 0, $context = '')
    {
		// Initialise variables.
		$recordId	= (int) $recordId;
		$ownerId	= (int) $ownerId;
		$user		= JFactory::getUser();
		$userId		= $user->get('id');
		
		// Check general component delete permission first.
		if ($user->authorise('core.delete', 'com_bwpostman')) {
			return true;
		}

        // Then check context delete permission.
		if ($user->authorise('core.delete', 'com_bwpostman.' . $context)) {
			return true;
		}

        // Next check item delete permission.
		if ($user->authorise('core.delete', 'com_bwpostman.' . $context . '.' . $recordId)) {
			return true;
		}

    	// Fallback on edit.own (only at context newsletter).
		if  ($context = 'newsletter') {
			// First test if the permission is available.
			if ($user->authorise('core.edit.own', 'com_bwpostman.'.$context . '.' . $recordId)) {
				// Test if the owner matches 'me'.
				if ($ownerId == $userId) return true;
			}
		}
       	// Fallback on edit.own (only at context subscriber at frontend).
		if  ($context = 'subscriber') {
			// First test if the permission is available.
			if ($user->authorise('core.edit.own', 'com_bwpostman.'.$context . '.' . $recordId)) {
				// Test if the owner matches 'me'.
				if ($ownerId == $userId) return true;
			}
		}
		return false;
	}
 
   /**
     * Method to check if you can restore an archived record.
     *
     * @param    int		$recordId	The record to test.
     * @param    int		$ownerId	The user to test against.
     * @param    string		$context    The name of the context.
     *
     * @return    boolean
     * @since    1.6
     */
    public static function allowRestore($recordId = 0, $ownerId = 0, $context = '')
    {
		// Initialise variables.
		$recordId	= (int) $recordId;
		$ownerId	= (int) $ownerId;
		$user		= JFactory::getUser();
		$userId		= $user->get('id');
		
		// Check general component restore permission first.
		if ($user->authorise('core.restore', 'com_bwpostman')) {
			return true;
		}

        // Then check context restore permission.
		if ($user->authorise('core.restore', 'com_bwpostman.' . $context)) {
			return true;
		}

        // Next check item restore permission.
		if ($user->authorise('core.restore', 'com_bwpostman.' . $context . '.' . $recordId)) {
			return true;
		}

		// Fallback on edit.own.
		// First test if the permission is available.
		if ($user->authorise('core.edit.own', 'com_bwpostman.'.$context.'.' . $recordId)) {
			// Test if the owner matches 'me'.
			if ($ownerId == $userId) return true;
		}
		return false;
	}
 
	/**
	 * Method to get all published mailinglists
     *
     * @return    string
     * @since    0.9.8
     */
    public static function getMailinglistsWarning()
    {
    	$_db			= JFactory::getDbo();
    	$query			= $_db->getQuery(true);
    	$ml_published	='';
	
		// Get # of all published mailinglists
    	$query->select('COUNT(*)');
		$query->from($_db->quoteName('#__bwpostman_mailinglists'));
		$query->where($_db->quoteName('published') . ' = ' . (int) 1);
		$query->where($_db->quoteName('archive_flag') . ' = ' . (int) 0);
				
		$_db->setQuery($query);
		
		$ml_published = $_db->loadResult();
	
	    if ($ml_published <1){
	      JFactory::getApplication()->enqueueMessage(JText::_('COM_BWPOSTMAN_NL_WARNING_NO_PUBLISHED_MAILINGLIST'), 'warning');
	    }
	    unset($ml_published);
    }


	/**
	 * Check number of queue entries
	 * 
	 * @return	bool	true if there are entries in the queue, otherwise false
	 * 
	 * since 1.0.3
	 */
	static public function checkQueueEntries()
	{
		$_db	= JFactory::getDbo();
		$query	= $_db->getQuery(true);
		
		$query->select('COUNT(' . $_db->quoteName('id') . ')');
		$query->from($_db->quoteName('#__bwpostman_sendmailqueue'));
		
		$_db->setQuery($query);

		if ($_db->loadResult() > 0) {
			return true;
		}
		else {
			return false;
		}
	}

    /**
     * Method to get a captcha string
     *
     * @return    string
     * @since    0.9.8
     */
    public static function getCaptcha($mode = 1)
    {
		$zahl		= 1960;
		$no_spam	= '';
		if ($mode == 1) {
			$no_spam = (date("dmy", time())) * $zahl;
		}
		if ($mode == 2) {
			if (date('H', time())=='00') {
				$no_spam = (date("dmy", time()-86400)) * $zahl;
			}
		}
		return $no_spam;
    }
 
	/**
	 *	Captcha Bild
	 *
	 *	Systemvoraussetzung:
	 *	Linux, Windows
	 * 	PHP 4 >= 4.0.0-RC2 , PHP 5
	 *	GD-Bibliothek ( > gd-1.6 )
	 *	FreeType-Bibliothek
	 *
	 *
	 * 	LICENSE: GNU General Public License (GPL)
	 *	This program is free software; you can redistribute it and/or modify
	 *	it under the terms of the GNU General Public License version 2,
	 *	as published by the Free Software Foundation.
	 *
	 *	@category   Captcha
	 *	@author     Damir Enseleit <info@selfphp.de>
	 *	@copyright  2001-2006 SELFPHP
	 *	@version    $Id: captcha.php,v 0.10 2006/04/07 13:15:30 des1 Exp $
	 *	@link       http://www.selfphp.de
	 */
	
	/**
	 * Erzeugt die Rechenaufgabe
	 *
	 * @param		resource	$im				Zeiger auf das Bild
	 * @param		integer		$size			Schriftgrösse
	 * @param		string		$fileTTF		Pfad zur True-Type-Datei
	 * @param		integer		$imgHeight		Bildhöhe
	 *
	 * @return		string		Gibt die Rechenaufgabe als String für den Dateinamen wieder			
	 */
	
	static public function showCaptcha() {
		function mathCaptcha($im,$size,$fileTTF,$imgHeight)
		{
			$math = range(0,9);
			shuffle($math);
			
			$mix = range(0,120);
			shuffle($mix);
			
			$color = imagecolorallocate($im,$mix[0],$mix[1],$mix[2]);
		
			$text		= "$math[0] + $math[1]";
			$fileName	= $math[0] + $math[1];
			
			imagettftext($im, $size, 0, 5, 25, $color, $fileTTF,$text);
			
			return $fileName;
		}

	// TTF-Schrift
	// Sie sollten hier unbedingt den absoluten Pfad angeben, da ansonsten
	// eventuell die TTF-Datei nicht eingebunden werden kann!
	$fileTTF =  JPATH_COMPONENT_SITE.'/assets/ttf/style.ttf';
	
	// Verzeichnis für die Captcha-Bilder (muss Schreibrechte besitzen!)
	// Ausserdem sollten in diesem Ordner nur die Bilder gespeichert werden
	// da das Programm in regelmaessigen Abstaenden dieses leert!
	// Kein abschliessenden Slash benutzen!
	$captchaDir = JPATH_COMPONENT_SITE.'/assets/capimgdir';
	
	// Schriftgröße
	$size = 20;
	
	// Schriftgröße Rechenaufgabe
	$sizeMath = 20;
	
	//Bildgroesse
	$imgWidth = 80;//200
	$imgHeight = 30;//80
	
	header("Content-type: image/png");
	$im = @imagecreate($imgWidth, $imgHeight)
	  or die("GD! Initialisierung fehlgeschlagen");
	$color = imagecolorallocate($im,255,255,255);
	imagefill($im,0,$imgWidth,$color);
	$fileName = mathCaptcha($im,$sizeMath,$fileTTF,$imgHeight);
	
	// Uebermittelter Hash-Wert ueberpruefen
	if(!preg_match('/^[a-f0-9]{32}$/',$_GET['codeCaptcha']))
		$_GET['codeCaptcha'] = md5(microtime());
	
	// Image speichern	
	imagepng($im,$captchaDir.'/'.$_GET['codeCaptcha'].'_'.$fileName.'.png');
	imagedestroy($im);
	// Bild ausgeben
	readfile(JURI::base().'components/com_bwpostman/assets/capimgdir/'.$_GET['codeCaptcha'].'_'.$fileName.'.png');
	}


	/**
	 *	Captcha Bild Überprüfung
	 *
	 *	Systemvoraussetzung:
	 *	Linux, Windows
	 * 	PHP 4 >= 4.0.0-RC2 , PHP 5
	 *	GD-Bibliothek (> gd-1.6)
	 *	FreeType-Bibliothek
	 *
	 *	Prüft ein Captcha-Bild
	 *
	 * 	LICENSE: GNU General Public License (GPL)
	 *	This program is free software; you can redistribute it and/or modify
	 *	it under the terms of the GNU General Public License version 2,
	 *	as published by the Free Software Foundation.
	 *
	 *	@category   Captcha
	 *	@author     Damir Enseleit <info@selfphp.de>
	 *	@copyright  2001-2006 SELFPHP
	 *	@version    $Id: captcha_check.php,v 0.10 2006/04/07 13:15:30 des1 Exp $
	 *	@link       http://www.selfphp.de
	 *
	 * @param		string		$codeCaptcha		Hash-Wert
	 * @param		string		$stringCaptcha		Eingabe durch den User
	 * @param		string		$dir				Das Verzeichnis mit den Captcha-Bilder
	 * @param		integer		$delFile			Die Zeit in Minuten, nachdem ein Captcha-Bild gelöscht wird
	 *
	 * @return		bool		TRUE/FALSE			
	 */
	public static function CheckCaptcha($codeCaptcha,$stringCaptcha,$dir,$delFile=5)
	{
		// Setzt den Check erst einmal auf FALSE
		$captchaTrue = FALSE;
		
		// Übergebene Hash-Variable überprüfen
		if(!preg_match('/^[a-f0-9]{32}$/',$codeCaptcha))
			return FALSE;
		
		// Übergebene Captcha-Variable überprüfen
		if(!preg_match('/^[a-zA-Z0-9]{1,6}$/',$stringCaptcha))
			return FALSE;
		
		$handle = @opendir($dir);
		while (false !== ($file = readdir($handle))) 
		  {
			if (preg_match("=^\.{1,2}$=", $file))
		    {
		       continue;
		    }
		       
		    if (is_dir($dir.$file))
		    {
		       continue;
		    }
		    else
		    {
		    	$lastTime = ceil((time() - filemtime($dir.$file)) / 60);

		    	if($lastTime > $delFile)
		    	{
		    		if ($file != 'index.html') unlink($dir.$file);
		    	}
		    	else{
		    		if(strtolower($file) == strtolower($codeCaptcha.'_'.$stringCaptcha.'.png'))
		    		{
		    			$captchaTrue = TRUE;
		    		}  
		    		if (preg_match("=^$codeCaptcha=i", $file))
		    		{
		    			if ($file != 'index.html') unlink($dir.$file); 
		    		}
		     	}
		    }
		}	
	
		@closedir($handle);
		
		if ($captchaTrue)
			return TRUE;
		else
			return FALSE;
	}

	/**
	 * Load BwPostman language file
	 *
	 * Helper function for external modules and plugins to load the main BwPostman language file(s)
	 *
	 */
	public static function loadLanguage($file = 'com_bwpostman', $client = 'site') {
		static $loaded = array();
//		BWPOSTMAN_PROFILER ? BwPostmanProfiler::instance()->start('function '.__CLASS__.'::'.__FUNCTION__.'()') : null;

		if ($client == 'site') {
			$lookup1 = JPATH_SITE;
			$lookup2 = BWPOSTMAN_PATH_SITE;
		}
		else {
			$client = 'admin';
			$lookup1 = JPATH_ADMINISTRATOR;
			$lookup2 = BWPOSTMAN_PATH_ADMIN;
		}
		if (empty($loaded["{$client}/{$file}"])) {
			$lang = JFactory::getLanguage();
			if (version_compare(JVERSION, '1.6','<')) {
				// Joomla 1.5 hack to make languages to load faster
				if ($lang->getTag() != 'en-GB' && !JDEBUG && !$lang->getDebug()
						&& !BwPostmanFactory::getConfig()->get('debug') && BwPostmanFactory::getConfig()->get('fallback_english')) {
					$filename = JLanguage::getLanguagePath($lookup2, 'en-GB')."/en-GB.{$file}.ini";
					$loaded[$file] = self::parseLanguage($lang, $filename);
				}
				$filename = JLanguage::getLanguagePath($lookup1, $lang->_lang)."/{$lang->_lang}.{$file}.ini";
				$loaded[$file] = self::parseLanguage($lang, $filename);
				if (!$loaded[$file]) {
					$filename = JLanguage::getLanguagePath($lookup2, $lang->_lang)."/{$lang->_lang}.{$file}.ini";
					$loaded[$file] = self::parseLanguage($lang, $filename);
				}
				if (!$loaded[$file]) {
					$filename = JLanguage::getLanguagePath($lookup1, $lang->_default)."/{$lang->_default}.{$file}.ini";
					$loaded[$file] = self::parseLanguage($lang, $filename);
				}
				if (!$loaded[$file]) {
					$filename = JLanguage::getLanguagePath($lookup2, $lang->_default)."/{$lang->_default}.{$file}.ini";
					$loaded[$file] = self::parseLanguage($lang, $filename);
				}
			}
			else {
				$english = false;
				if ($lang->getTag() != 'en-GB' && !JDEBUG && !$lang->getDebug()
//						&& !BwPostmanFactory::getConfig()->get('debug') 
//						&& BwPostmanFactory::getConfig()->get('fallback_english')
					) {
					$lang->load($file, $lookup2, 'en-GB', true, false);
					$english = true;
				}
				$loaded[$file] = $lang->load($file, $lookup1, null, $english, false)
					|| $lang->load($file, $lookup2, null, $english, false)
					|| $lang->load($file, $lookup1, $lang->getDefault(), $english, false)
					|| $lang->load($file, $lookup2, $lang->getDefault(), $english, false);
			}
		}
//		BWPOSTMAN_PROFILER ? BwPostmanProfiler::instance()->stop('function '.__CLASS__.'::'.__FUNCTION__.'()') : null;
		return $loaded[$file];
}

	protected static function parseLanguage($lang, $filename) {
		if (!file_exists($filename)) return false;

		$version = phpversion();

		// Capture hidden PHP errors from the parsing.
		$php_errormsg = null;
		$track_errors = ini_get('track_errors');
		ini_set('track_errors', true);

		if ($version >= '5.3.1') {
			$contents = file_get_contents($filename);
			$contents = str_replace('_QQ_', '"\""', $contents);
			$strings = @parse_ini_string($contents);
		}
		else {
			$strings = @parse_ini_file($filename);

			if ($version == '5.3.0' && is_array($strings)) {
				foreach ($strings as $key => $string) {
					$strings[$key] = str_replace('_QQ_', '"', $string);
				}
			}
		}

		// Restore error tracking to what it was before.
		ini_set('track_errors', $track_errors);

		if (!is_array($strings)) {
			$strings = array();
		}

		$lang->_strings = array_merge($lang->_strings, $strings);
		return !empty($strings);
	}
}